<?php
namespace FileScan;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class FSC_Admin {

    private static $instance = null;

    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action( 'admin_menu', [ $this, 'register_menu' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ] );
        add_action( 'wp_ajax_fsc_search_code', [ $this, 'ajax_search_code' ] );
    }

    /**
     * Register menu item under Tools.
     */
    public function register_menu() {
        add_management_page(
            __( 'File Scan Code Search', 'file-scan-code-search' ),
            __( 'File Scan', 'file-scan-code-search' ),
            'manage_options',
            'file-scan-code-search',
            [ $this, 'render_admin_page' ]
        );
    }

    /**
     * Enqueue CSS and JS only on our page.
     *
     * @param string $hook Hook.
     */
    public function enqueue_assets( $hook ) {
        if ( 'tools_page_file-scan-code-search' !== $hook ) {
            return;
        }

        wp_enqueue_style(
            'fsc-admin',
            FSC_PLUGIN_URL . 'admin/css/fsc-admin.css',
            [],
            FSC_VERSION
        );

        wp_enqueue_script(
            'fsc-admin',
            FSC_PLUGIN_URL . 'admin/js/fsc-admin.js',
            [ 'jquery' ],
            FSC_VERSION,
            true
        );

        wp_localize_script(
            'fsc-admin',
            'FSC_Admin',
            [
                'ajax_url' => admin_url( 'admin-ajax.php' ),
                'nonce'    => wp_create_nonce( 'fsc_admin_nonce' ),
                'site_url' => get_site_url(),
            ]
        );
    }

    /**
     * Render admin page.
     */
    public function render_admin_page() {
        $locations = $this->get_detected_locations();

        $fscs_allowed_tabs = array( 'search', 'advanced', 'help' );
        $fscs_tab          = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'search'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only tab switch.
        if ( ! in_array( $fscs_tab, $fscs_allowed_tabs, true ) ) {
            $fscs_tab = 'search';
        }

        $fscs_page_slug = 'file-scan-code-search';
        $fscs_base_url  = admin_url( 'tools.php?page=' . $fscs_page_slug );

        $fscs_tab_urls = array(
            'search'   => add_query_arg( array( 'tab' => 'search' ), $fscs_base_url ),
            'advanced' => add_query_arg( array( 'tab' => 'advanced' ), $fscs_base_url ),
            'help'     => add_query_arg( array( 'tab' => 'help' ), $fscs_base_url ),
        );

        include FSC_PLUGIN_DIR . 'admin/partials/fsc-admin-display.php';
    }

    /**
     * Handle AJAX search.
     */
    public function ajax_search_code() {
        check_ajax_referer( 'fsc_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( __( 'Unauthorized', 'file-scan-code-search' ) );
        }

        $search_text    = isset( $_POST['search_text'] ) ? sanitize_text_field( wp_unslash( $_POST['search_text'] ) ) : '';
        $base_dir       = isset( $_POST['base_dir'] ) ? sanitize_key( wp_unslash( $_POST['base_dir'] ) ) : '';
        $extensions     = isset( $_POST['extensions'] ) ? sanitize_text_field( wp_unslash( $_POST['extensions'] ) ) : '';
        $case_sensitive = isset( $_POST['case_sensitive'] ) ? $this->to_bool( sanitize_text_field( wp_unslash( $_POST['case_sensitive'] ) ) ) : false;
        $regex          = isset( $_POST['regex'] ) ? $this->to_bool( sanitize_text_field( wp_unslash( $_POST['regex'] ) ) ) : false;

        if ( '' === $search_text ) {
            wp_send_json_error( __( 'Search text empty', 'file-scan-code-search' ) );
        }

        // Resolve the search path.
        $locations = $this->get_detected_locations();

        if ( 'custom' === $base_dir ) {
            $custom_raw  = isset( $_POST['custom_path'] ) ? sanitize_text_field( wp_unslash( $_POST['custom_path'] ) ) : '';
            $custom_real = $this->validate_custom_path( $custom_raw );

            if ( false === $custom_real ) {
                wp_send_json_error( __( 'Invalid custom path. Try selecting Site home, or enable advanced mode to scan outside WordPress.', 'file-scan-code-search' ) );
            }

            $search_path = $custom_real;
        } elseif ( isset( $locations[ $base_dir ] ) ) {
            if ( empty( $locations[ $base_dir ]['enabled'] ) ) {
                wp_send_json_error( __( 'This location is disabled', 'file-scan-code-search' ) );
            }
            $search_path = $locations[ $base_dir ]['path'];
        } else {
            wp_send_json_error( __( 'Invalid directory', 'file-scan-code-search' ) );
        }

        // Build extension list.
        $exts = array_filter( array_map( 'trim', explode( ',', $extensions ) ) );
        if ( empty( $exts ) ) {
            $exts = [ 'php', 'js', 'css' ];
        }

        // Safety limits for large scans.
        $limits = [
            'max_files'   => 12000,
            'max_results' => 800,
            'max_seconds' => 20,
        ];

        // Root scans can be slower. Increase limits to avoid missing matches.
        if ( in_array( $base_dir, [ 'wp_root', 'site_home' ], true ) ) {
            $limits['max_files']   = 60000;
            $limits['max_seconds'] = 60;
        }

        $results = $this->perform_search( $search_text, $search_path, $exts, $case_sensitive, $regex, $limits, $max_depth );

        wp_send_json_success( $results );
    }

    /**
     * Get detected directory locations.
     *
     * @return array<string, array{label:string,path:string,enabled:bool}>
     */
    /**
     * Get the site home path safely (admin-only).
     *
     * @return string|false
     */
    private function get_home_path_safe() {
        if ( ! is_admin() ) {
            return false;
        }

        if ( ! function_exists( 'get_home_path' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        if ( function_exists( 'get_home_path' ) ) {
            $p = get_home_path();
            if ( is_string( $p ) && '' !== trim( $p ) ) {
                return wp_normalize_path( $p );
            }
        }

        // Fallback: document root if present.
        if ( ! empty( $_SERVER['DOCUMENT_ROOT'] ) ) {
            $doc = wp_normalize_path( sanitize_text_field( wp_unslash( (string) $_SERVER['DOCUMENT_ROOT'] ) ) );
            if ( is_dir( $doc ) ) {
                return $doc;
            }
        }

        return false;
    }

function get_detected_locations() {
        $locations = [];

        $locations['wp_root'] = [
            'label'   => __( 'WordPress root', 'file-scan-code-search' ),
            'path'    => trailingslashit( ABSPATH ),
            'enabled' => true,
        ];
        // Site home can differ from ABSPATH when WordPress is installed in a subdirectory.
        $home_path = $this->get_home_path_safe();
        if ( $home_path && is_dir( $home_path ) ) {
            $locations['site_home'] = [
                'label'   => __( 'Site home', 'file-scan-code-search' ),
                'path'    => trailingslashit( $home_path ),
                'enabled' => true,
            ];
        }

        $locations['wp_admin'] = [
            'label'   => __( 'wp-admin', 'file-scan-code-search' ),
            'path'    => trailingslashit( ABSPATH . 'wp-admin' ),
            'enabled' => is_dir( ABSPATH . 'wp-admin' ),
        ];

        $locations['wp_includes'] = [
            'label'   => __( 'wp-includes', 'file-scan-code-search' ),
            'path'    => trailingslashit( ABSPATH . 'wp-includes' ),
            'enabled' => is_dir( ABSPATH . 'wp-includes' ),
        ];



        $locations['wp_content'] = [
            'label'   => __( 'wp-content', 'file-scan-code-search' ),
            'path'    => trailingslashit( WP_CONTENT_DIR ),
            'enabled' => true,
        ];

        $locations['plugins'] = [
            'label'   => __( 'Plugins', 'file-scan-code-search' ),
            'path'    => trailingslashit( WP_PLUGIN_DIR ),
            'enabled' => true,
        ];


// Individual plugin folders (fast targeting).
if ( is_dir( WP_PLUGIN_DIR ) && is_readable( WP_PLUGIN_DIR ) ) {
    $plugin_dirs = @scandir( WP_PLUGIN_DIR );
    if ( is_array( $plugin_dirs ) ) {
        $count = 0;
        foreach ( $plugin_dirs as $pd ) {
            if ( $pd === '.' || $pd === '..' ) {
                continue;
            }
            $full = trailingslashit( WP_PLUGIN_DIR ) . $pd;
            if ( is_dir( $full ) && is_readable( $full ) ) {
                $key = 'plugin_' . sanitize_key( $pd );
                $locations[ $key ] = array(
                    /* translators: %s is a plugin folder name. */
                    'label'   => sprintf( __( 'Plugin: %s', 'file-scan-code-search' ), $pd ),
                    'path'    => trailingslashit( $full ),
                    'enabled' => true,
                );
                $count++;
                if ( $count >= 50 ) {
                    break;
                }
            }
        }
    }
}

        if ( defined( 'WPMU_PLUGIN_DIR' ) && is_string( WPMU_PLUGIN_DIR ) && is_dir( WPMU_PLUGIN_DIR ) ) {
            $locations['mu_plugins'] = [
                'label'   => __( 'Must-use plugins', 'file-scan-code-search' ),
                'path'    => trailingslashit( WPMU_PLUGIN_DIR ),
                'enabled' => true,
            ];
        }

        $theme_root = get_theme_root();
        if ( $theme_root && is_dir( $theme_root ) ) {
            $locations['themes'] = [
                'label'   => __( 'Themes', 'file-scan-code-search' ),
                'path'    => trailingslashit( $theme_root ),
                'enabled' => true,
            ];


// Individual theme folders (fast targeting).
if ( isset( $locations['themes']['path'] ) && is_dir( $locations['themes']['path'] ) ) {
    $theme_dirs = @scandir( untrailingslashit( $locations['themes']['path'] ) );
    if ( is_array( $theme_dirs ) ) {
        $count = 0;
        foreach ( $theme_dirs as $td ) {
            if ( $td === '.' || $td === '..' ) {
                continue;
            }
            $full = trailingslashit( untrailingslashit( $locations['themes']['path'] ) ) . $td;
            if ( is_dir( $full ) && is_readable( $full ) ) {
                $key = 'theme_' . sanitize_key( $td );
                $locations[ $key ] = array(
                    /* translators: %s is a theme folder name. */
                    'label'   => sprintf( __( 'Theme: %s', 'file-scan-code-search' ), $td ),
                    'path'    => trailingslashit( $full ),
                    'enabled' => true,
                );
                $count++;
                if ( $count >= 50 ) {
                    break;
                }
            }
        }
    }
}
        }

        $upload_dir = wp_get_upload_dir();
        if ( ! empty( $upload_dir['basedir'] ) && is_dir( $upload_dir['basedir'] ) ) {
            $locations['uploads'] = [
                'label'   => __( 'Uploads', 'file-scan-code-search' ),
                'path'    => trailingslashit( $upload_dir['basedir'] ),
                'enabled' => true,
            ];


// Additional common locations.
if ( defined( 'WP_LANG_DIR' ) && is_string( WP_LANG_DIR ) && is_dir( WP_LANG_DIR ) ) {
    $locations['languages'] = [
        'label'   => __( 'Languages', 'file-scan-code-search' ),
        'path'    => trailingslashit( WP_LANG_DIR ),
        'enabled' => true,
    ];
}

$stylesheet_dir = function_exists( 'get_stylesheet_directory' ) ? get_stylesheet_directory() : '';
if ( $stylesheet_dir && is_dir( $stylesheet_dir ) ) {
    $locations['active_theme'] = [
        'label'   => __( 'Active theme', 'file-scan-code-search' ),
        'path'    => trailingslashit( $stylesheet_dir ),
        'enabled' => true,
    ];
}

$template_dir = function_exists( 'get_template_directory' ) ? get_template_directory() : '';
if ( $template_dir && is_dir( $template_dir ) && $template_dir !== $stylesheet_dir ) {
    $locations['parent_theme'] = [
        'label'   => __( 'Parent theme', 'file-scan-code-search' ),
        'path'    => trailingslashit( $template_dir ),
        'enabled' => true,
    ];
}

        }

        // Advanced locations (outside WordPress) are disabled by default.
        $allow_outside = defined( 'FSC_ALLOW_OUTSIDE_WP' ) && FSC_ALLOW_OUTSIDE_WP;

        $account_root = dirname( untrailingslashit( ABSPATH ) );
        if ( $account_root && is_dir( $account_root ) ) {
            $locations['account_root'] = [
                'label'   => __( 'Account root (advanced)', 'file-scan-code-search' ),
                'path'    => trailingslashit( $account_root ),
                'enabled' => (bool) $allow_outside,
            ];
        }

        // If open_basedir is set, use it as a hint for safe roots.
        $open_basedir = ini_get( 'open_basedir' );
        if ( is_string( $open_basedir ) && '' !== trim( $open_basedir ) ) {
            $parts = preg_split( '#[;:]#', $open_basedir );
            $parts = array_filter( array_map( 'trim', (array) $parts ) );
            $i     = 0;

            foreach ( $parts as $p ) {
                if ( $i >= 5 ) {
                    break;
                }
                if ( ! $p || ! is_dir( $p ) ) {
                    continue;
                }

                $locations[ 'open_basedir_' . $i ] = [
                    // translators: %d is the index of an allowed root path.
                    'label'   => sprintf( __( 'Server allowed root %d (advanced)', 'file-scan-code-search' ), $i + 1 ),
                    'path'    => trailingslashit( $p ),
                    'enabled' => (bool) $allow_outside,
                ];

                $i++;
            }
        }

        
        $temp_dir = function_exists( 'sys_get_temp_dir' ) ? sys_get_temp_dir() : '';
        if ( $temp_dir && is_dir( $temp_dir ) ) {
            $locations['temp_dir'] = [
                'label'   => __( 'System temp directory (advanced)', 'file-scan-code-search' ),
                'path'    => trailingslashit( $temp_dir ),
                'enabled' => (bool) $allow_outside,
            ];
        }

// Server root.
        $locations['server_root'] = [
            'label'   => __( 'Server root (advanced)', 'file-scan-code-search' ),
            'path'    => trailingslashit( DIRECTORY_SEPARATOR ),
            'enabled' => (bool) $allow_outside,
        ];

        

// Common server directories (advanced). These are only enabled when FSC_ALLOW_OUTSIDE_WP is true.
$common_dirs = array(
    'home_dir' => array( 'label' => __( 'Home directory (/home) (advanced)', 'file-scan-code-search' ), 'path' => '/home' ),
    'var_www'  => array( 'label' => __( 'Web root (/var/www) (advanced)', 'file-scan-code-search' ), 'path' => '/var/www' ),
    'var_dir'  => array( 'label' => __( 'System folder (/var) (advanced)', 'file-scan-code-search' ), 'path' => '/var' ),
    'tmp_dir'  => array( 'label' => __( 'Temp folder (/tmp) (advanced)', 'file-scan-code-search' ), 'path' => '/tmp' ),
);

foreach ( $common_dirs as $k => $d ) {
    if ( empty( $d['path'] ) ) {
        continue;
    }
    $p = wp_normalize_path( $d['path'] );
    if ( is_dir( $p ) ) {
        $locations[ $k ] = array(
            'label'   => $d['label'],
            'path'    => trailingslashit( $p ),
            'enabled' => (bool) $allow_outside,
        );
    }
}

return apply_filters( 'file_scan_code_search_detected_locations', $locations );
    }

    /**
     * Validate a custom path against allowed roots.
     *
     * By default, only paths inside ABSPATH are allowed. To allow scanning outside, define:
     * define( 'FSC_ALLOW_OUTSIDE_WP', true );
     *
     * @param string $custom_path User input path.
     * @return string|false Real path or false.
     */
    private function validate_custom_path( $custom_path ) {
        $custom_path = wp_normalize_path( (string) $custom_path );
        $custom_path = trim( $custom_path );

        if ( '' === $custom_path ) {
            return false;
        }

        $real = realpath( $custom_path );
        if ( false === $real ) {
            return false;
        }

        $real = wp_normalize_path( $real );

        // If a file path is provided, use its parent directory.
        if ( is_file( $real ) ) {
            $real = wp_normalize_path( dirname( $real ) );
        }

        // Always allow inside ABSPATH.

        // Also allow inside site home when WordPress is installed in a subdirectory.
        $home = $this->get_home_path_safe();
        if ( $home ) {
            $home = wp_normalize_path( trailingslashit( $home ) );
            if ( 0 === strpos( $real, $home ) ) {
                return $real;
            }
        }

        $abs = wp_normalize_path( trailingslashit( ABSPATH ) );
        if ( 0 === strpos( $real, $abs ) ) {
            return $real;
        }

        // Outside roots only when enabled.
        $allow_outside = defined( 'FSC_ALLOW_OUTSIDE_WP' ) && FSC_ALLOW_OUTSIDE_WP;
        if ( ! $allow_outside ) {
            return false;
        }

        // Respect open_basedir if set.
        $open_basedir = ini_get( 'open_basedir' );
        if ( is_string( $open_basedir ) && '' !== trim( $open_basedir ) ) {
            $parts   = preg_split( '#[;:]#', $open_basedir );
            $parts   = array_filter( array_map( 'trim', (array) $parts ) );
            $allowed = false;

            foreach ( $parts as $p ) {
                if ( ! $p ) {
                    continue;
                }

                $p = wp_normalize_path( trailingslashit( $p ) );
                if ( 0 === strpos( $real, $p ) ) {
                    $allowed = true;
                    break;
                }
            }

            if ( ! $allowed ) {
                return false;
            }
        }

        if ( ! is_dir( $real ) || ! is_readable( $real ) ) {
            return false;
        }

        return $real;
    }

    /**
     * Perform the file scan.
     *
     * @param string $needle Needle.
     * @param string $path Base path.
     * @param array  $extensions Extensions.
     * @param bool   $case_sensitive Case sensitive.
     * @param bool   $regex Regex mode.
     * @param array  $limits Limits.
     * @return array
     */
    private function perform_search( $needle, $path, array $extensions, $case_sensitive = false, $regex = false, array $limits = [], $max_depth = 0 ) {
        $results = [];
        $start   = microtime( true );
        $files   = 0;

        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator( $path, \FilesystemIterator::SKIP_DOTS ),
            \RecursiveIteratorIterator::SELF_FIRST
        );

        // Depth control: 0 means unlimited.
        $depth = (int) $max_depth;
        $iterator->setMaxDepth( ( 0 === $depth ) ? -1 : $depth );

        $pattern_extensions = '/\.(' . implode( '|', array_map( 'preg_quote', $extensions ) ) . ')$/i';

        foreach ( $iterator as $file ) {
            if ( ! $file instanceof \SplFileInfo ) {
                continue;
            }

            $files++;

            if ( ! empty( $limits['max_files'] ) && $files > (int) $limits['max_files'] ) {
                break;
            }

            if ( ! empty( $limits['max_seconds'] ) && ( microtime( true ) - $start ) > (float) $limits['max_seconds'] ) {
                break;
            }

            $file_path = $file->getPathname();

            if ( ! preg_match( $pattern_extensions, $file_path ) ) {
                continue;
            }

            // Skip large files (>1 MB).
            if ( $file->getSize() > 1024 * 1024 ) {
                continue;
            }

            $lines = @file( $file_path, FILE_IGNORE_NEW_LINES );
            if ( ! $lines ) {
                continue;
            }

            foreach ( $lines as $number => $line ) {
                $match = false;

                if ( $regex ) {
                    $flags   = $case_sensitive ? '' : 'i';
                    $pattern = '/' . str_replace( '/', '\\/', $needle ) . '/' . $flags;

                    if ( false === @preg_match( $pattern, '' ) ) {
                        wp_send_json_error( __( 'Invalid regex pattern', 'file-scan-code-search' ) );
                    }

                    $match = ( 1 === preg_match( $pattern, $line ) );
                } else {
                    $match = $case_sensitive ? ( false !== strpos( $line, $needle ) ) : ( false !== stripos( $line, $needle ) );
                }

                if ( $match ) {
                    $display_path = ( 0 === strpos( wp_normalize_path( $file_path ), wp_normalize_path( ABSPATH ) ) )
                        ? str_replace( ABSPATH, '', $file_path )
                        : $file_path;

                    $results[] = [
                        'file'    => $display_path,
                        'line'    => $number + 1,
                        'snippet' => $this->highlight_match( trim( (string) $line ), $needle, $case_sensitive, $regex ),
                    ];

                    if ( ! empty( $limits['max_results'] ) && count( $results ) >= (int) $limits['max_results'] ) {
                        break 2;
                    }
                }
            }
        }

        return $results;
    }

    /**
     * Highlight matches in a snippet.
     *
     * @param string $line Line.
     * @param string $needle Needle.
     * @param bool   $case_sensitive Case sensitive.
     * @param bool   $regex Regex.
     * @return string
     */
    private function highlight_match( $line, $needle, $case_sensitive, $regex ) {
        if ( $regex ) {
            $flags   = $case_sensitive ? '' : 'i';
            $pattern = '/' . str_replace( '/', '\\/', $needle ) . '/' . $flags;
            return preg_replace( $pattern, '<mark>$0</mark>', esc_html( $line ) );
        }

        $escaped = preg_quote( $needle, '/' );
        $flags   = $case_sensitive ? '' : 'i';
        return preg_replace( '/' . $escaped . '/' . $flags, '<mark>$0</mark>', esc_html( $line ) );
    }

    /**
     * Convert request value to bool.
     *
     * @param mixed $value Value.
     * @return bool
     */
    private function to_bool( $value ) {
        $value = is_string( $value ) ? strtolower( trim( $value ) ) : $value;
        return in_array( $value, [ true, 1, '1', 'true', 'yes', 'on' ], true );
    }
}
